<?php
/*
Plugin Name:       ImpressumAPI
Plugin URI:        https://github.com/nikolajewski09media/ImpressumAPI/releases
Description:       Short_codes: [impressum09] oder [datenschutz09] oder [handynummer09] oder [handynummer09 fancy="true"] oder [callNowButton09 bg-color="blue" font-color="pink" link-color="pink"] wobei ihr die Farben frei wählen könnt
Author: Herbert Nikolajewski
Version:           1.1.4
Author:            Herby Nikolajewski
GitHub Plugin URI: https://github.com/nikolajewski09media/ImpressumAPI
GitHub Branch:     main
*/

/**
 * Klasse für die Plugin-Update-Logik von einer externen URL.
 * Basiert auf dem Prinzip des "Plugin Update Checkers"
 */
if ( ! class_exists( 'ImpressumAPI_Plugin_Updater' ) ) {
    class ImpressumAPI_Plugin_Updater {

        private $plugin_file;
        private $update_url;
        private $current_version;
        private $plugin_slug;
        private $version_info_url;

        /**
         * Konstruktor.
         *
         * @param string $plugin_file Der vollständige Pfad zur Haupt-Plugin-Datei.
         * @param string $update_url Die URL zur ZIP-Datei des Plugins.
         * @param string $version_info_url Die URL zu einer JSON-Datei mit Versionsinformationen.
         */
        public function __construct( $plugin_file, $update_url, $version_info_url ) {
            $this->plugin_file      = $plugin_file;
            $this->update_url       = $update_url;
            $this->version_info_url = $version_info_url;

            $plugin_data            = get_plugin_data( $this->plugin_file );
            $this->current_version  = $plugin_data['Version'];
            $this->plugin_slug      = plugin_basename( $this->plugin_file );

            add_filter( 'pre_set_site_transient_update_plugins', array( $this, 'check_for_update' ) );
            add_filter( 'plugins_api', array( $this, 'plugins_api_filter' ), 10, 3 );
        }

        /**
         * Holt die neuesten Versionsinformationen von der externen JSON-Datei.
         *
         * @return object|null Die Versionsinformationen oder null bei Fehler.
         */
        private function get_remote_version_info() {
            $response = wp_safe_remote_get( $this->version_info_url, array(
                'timeout' => 10, // Timeout von 10 Sekunden
                'headers' => array(
                    'Accept'     => 'application/json',
                    'User-Agent' => 'WordPress/' . get_bloginfo( 'version' ) . '; ' . get_bloginfo( 'url' ),
                ),
            ) );

            if ( is_wp_error( $response ) || wp_remote_retrieve_response_code( $response ) !== 200 ) {
                return null;
            }

            $body = wp_remote_retrieve_body( $response );
            $data = json_decode( $body );

            if ( ! is_object( $data ) || ! isset( $data->version ) ) {
                return null;
            }

            return $data;
        }

        /**
         * Überprüft, ob ein Update verfügbar ist.
         *
         * @param object $transient Das Transient-Objekt von WordPress.
         * @return object Das modifizierte Transient-Objekt.
         */
        public function check_for_update( $transient ) {
            if ( empty( $transient->checked ) ) {
                return $transient;
            }

            $remote_info = $this->get_remote_version_info();

            if ( $remote_info && version_compare( $remote_info->version, $this->current_version, '>' ) ) {
                $response_obj = new stdClass();
                $response_obj->slug        = $this->plugin_slug;
                $response_obj->new_version = $remote_info->version;
                $response_obj->url         = ! empty( $remote_info->plugin_uri ) ? $remote_info->plugin_uri : 'https://github.com/nikolajewski09media/ImpressumAPI'; // Fallback
                $response_obj->package     = $this->update_url;

                $transient->response[ $this->plugin_slug ] = $response_obj;
            }

            return $transient;
        }

        /**
         * Liefert detaillierte Informationen über das Plugin für das Update-Modal.
         *
         * @param bool   $false  Immer false.
         * @param string $action Die angeforderte Aktion.
         * @param object $arg    Das Argument-Objekt.
         * @return object|bool Das Plugin-Info-Objekt oder false.
         */
        public function plugins_api_filter( $false, $action, $arg ) {
            if ( 'plugin_information' !== $action || ! isset( $arg->slug ) || $arg->slug !== $this->plugin_slug ) {
                return $false;
            }

            $remote_info = $this->get_remote_version_info();
            if ( ! $remote_info ) {
                return $false;
            }

            $plugin_data = get_plugin_data( $this->plugin_file );

            $obj = new stdClass();
            $obj->name          = $plugin_data['Name'];
            $obj->slug          = $this->plugin_slug;
            $obj->version       = $remote_info->version;
            $obj->author        = $plugin_data['Author'];
            $obj->homepage      = $plugin_data['Plugin URI'];
            $obj->requires      = isset( $remote_info->requires ) ? $remote_info->requires : '5.0'; // Mindest WP Version
            $obj->tested        = isset( $remote_info->tested ) ? $remote_info->tested : '6.5';   // Getestet bis WP Version
            $obj->last_updated  = isset( $remote_info->last_updated ) ? $remote_info->last_updated : date( 'Y-m-d H:i:s' );
            $obj->download_link = $this->update_url;
            $obj->trunk         = $this->update_url; // Für direkte Installation
            $obj->sections      = array(
                'description' => $plugin_data['Description'],
                'changelog'   => isset( $remote_info->changelog ) ? wpautop( $remote_info->changelog ) : 'Keine Changelog-Informationen verfügbar.',
            );

            return $obj;
        }
    }

    // Initialisiere den Updater
    new ImpressumAPI_Plugin_Updater(
        __FILE__,
        'https://impressumapi.pages.dev/ImpressumAPI.zip', // URL zu Ihrer ZIP-Datei
        'https://impressumapi.pages.dev/latest_version.json' // URL zu Ihrer Versions-JSON-Datei
    );
}



 
/**
 * Fügt einen "Auf Aktualisierungen überprüfen"-Link zur Plugin-Liste hinzu.
 *
 * @param array  $links Die bestehenden Plugin-Metadaten-Links.
 * @param string $file  Der Pfad zur Haupt-Plugin-Datei.
 * @return array Die aktualisierten Links.
 */
function impressum09_add_update_check_link($links, $file) {
    if (plugin_basename(__FILE__) === $file) {
        $check_url = admin_url('update-core.php?force-check=1'); // Leert den Update-Cache und erzwingt einen Check
        $links['check_for_updates'] = '<a href="' . esc_url($check_url) . '">' . __('Auf Aktualisierungen überprüfen', 'textdomain') . '</a>';
    }
    return $links;
}
add_filter('plugin_row_meta', 'impressum09_add_update_check_link', 10, 2);





function impressum09_api_request($city)
{
    $api_url = 'https://sharp-noyce.82-165-243-146.plesk.page/api/collections/Impressen/records/';

    if ($city === "bottrop") {
        $api_url .= 'loma87xgnlr16ga';
    }
    if ($city === "regensburg") {
        $api_url .= 't2uoa5jbukefqn6';
    }

    // Korrekter Pfad zur JSON-Datei im Plugin-Verzeichnis
    $jsonFilePath = dirname(__FILE__) . '/impressum.json';

    $response = wp_safe_remote_get($api_url);
    $res_array = json_decode(wp_remote_retrieve_body($response), true); // Direkt den Body dekodieren

    if (count($res_array) === 0 || is_wp_error($response) || wp_remote_retrieve_response_code($response) >= 400) {
        if (file_exists($jsonFilePath)) {
            $jsonData = file_get_contents($jsonFilePath);
            $dataArray = json_decode($jsonData, true);
            return $dataArray;
        } else {
            return array(); // Leeres Array, wenn keine Daten und keine lokale Datei
        }
    }

    $data = wp_remote_retrieve_body($response);
    file_put_contents($jsonFilePath, $data); // $data ist bereits der JSON-String
    return $data;
}



function api_data_shortcode_impressum($atts) {
    $atts = shortcode_atts(
        array(
            'city' => 'regensburg',
        ),
        $atts
    );

    $city = $atts['city'];
    $api_data = impressum09_api_request($city);

    if ($api_data) {
        $data = json_decode($api_data, true); // Annahme: JSON-Daten werden empfangen

        $formatted_data = "";
        $formatted_data .= '<p style="text-align: left;">';
        if(str_contains($_SERVER['SERVER_NAME'], "seibel") === false){
            $formatted_data .= '<strong>' . $_SERVER['SERVER_NAME'] . '</strong>' . ' – ist ein Vermittlungsservice in Kooperation mit:';
        }
        $formatted_data .= '</p>';
        $formatted_data .= '<p style="text-align: left;">';
        $formatted_data .= '<strong>' . esc_html($data['Firma']) . '</strong>' . '<br />';
        $formatted_data .= esc_html($data['Strasse']) . " " . esc_html($data['Hausnummer']) . '<br />';
        $formatted_data .= esc_html($data['PLZ']) . " " . esc_html($data['Ort']) . '<br />';
        $formatted_data .= "E-Mail: " . esc_html($data['E_Mail']);
        $formatted_data .= '</p>';
        $formatted_data .= '<p style="text-align: left;">';
        $formatted_data .= 'Vertreten durch:' . '<br />';
        $formatted_data .= esc_html($data['CEO']) . '<br />';
        $formatted_data .= '</p>';
        if ($city === "regensburg") {
            $formatted_data .= '<p style="text-align: left;">';
            $formatted_data .= 'Persönlich haftende Gesellschafterin:' . '<br />';
            $formatted_data .= esc_html($data['Gesellschafter']) . '<br />';
            $formatted_data .= '</p>';

            $formatted_data .= '<p style="text-align: left;">';
            $formatted_data .= 'Gericht: ' . esc_html($data['Gericht']) . '<br />';
            $formatted_data .= 'Registernummer: ' . esc_html($data['HRNummer']) . '<br />';
            $formatted_data .= '</p>';

            $formatted_data .= '<p style="text-align: left;">';
            $formatted_data .= 'USt.-ID: ' . esc_html($data['UstID']) . '<br />';
            $formatted_data .= '</p>';
        }
        $formatted_data .= '<p style="text-align: left;">';
        $formatted_data .= 'Aufsichtsbehörde: ' . esc_html($data['Aufsichtsbehoerde']) . '<br />';
        $formatted_data .= '</p>';

        return $formatted_data;
    } else {
        return '<div>Fehler beim Abrufen der API-Daten.</div>';
    }
}
add_shortcode('impressum09', 'api_data_shortcode_impressum');

function api_data_shortcode_datenschutz($atts) {
    $atts = shortcode_atts(
        array(
            'city' => 'regensburg',
        ),
        $atts
    );

    $city = $atts['city'];
    $api_data = impressum09_api_request($city);

    if ($api_data) {
        $data = json_decode($api_data, true); // Annahme: JSON-Daten werden empfangen

        $formatted_data = "";
        $formatted_data .= '<p style="text-align: left;">';
        $formatted_data .= '<strong>' . esc_html($data['Firma']) . '</strong>' . '<br />';
        $formatted_data .= esc_html($data['Strasse']) . " " . esc_html($data['Hausnummer']) . '<br />';
        $formatted_data .= esc_html($data['PLZ']) . " " . esc_html($data['Ort']) . '<br />';
        $formatted_data .= "E-Mail: " . esc_html($data['E_Mail']);
        $formatted_data .= '</p>';
        $formatted_data .= '<p style="text-align: left;">';
        $formatted_data .= 'Vertreten durch:' . '<br />';
        $formatted_data .= esc_html($data['CEO']) . '<br />';
        $formatted_data .= '</p>';
        if ($city === "regensburg") {
            $formatted_data .= '<p style="text-align: left;">';
            $formatted_data .= 'Persönlich haftende Gesellschafterin:' . '<br />';
            $formatted_data .= esc_html($data['Gesellschafter']) . '<br />';
            $formatted_data .= '</p>';

            $formatted_data .= '<p style="text-align: left;">';
            $formatted_data .= 'Gericht: ' . esc_html($data['Gericht']) . '<br />';
            $formatted_data .= 'Registernummer: ' . esc_html($data['HRNummer']) . '<br />';
            $formatted_data .= '</p>';

            $formatted_data .= '<p style="text-align: left;">';
            $formatted_data .= 'USt.-ID: ' . esc_html($data['UstID']) . '<br />';
            $formatted_data .= '</p>';
        }
        $formatted_data .= '<p style="text-align: left;">';
        $formatted_data .= 'Aufsichtsbehörde: ' . esc_html($data['Aufsichtsbehoerde']) . '<br />';
        $formatted_data .= '</p>';

        return $formatted_data;
    } else {
        return '<div>Fehler beim Abrufen der API-Daten.</div>';
    }
}
add_shortcode('datenschutz09', 'api_data_shortcode_datenschutz');


function impressum09_api_request_telephone() {
    $api_url = 'https://sharp-noyce.82-165-243-146.plesk.page/api/collections/Handynummern/records?filter=(Domain=\'' . $_SERVER['SERVER_NAME'] . '\')';
    $jsonFilePath = dirname(__FILE__) . '/handynummer.json'; // Korrekter Pfad

    $response = wp_safe_remote_get($api_url);
    $res_array = json_decode(wp_remote_retrieve_body($response), true); // Direkt den Body dekodieren
    $res_array_items = isset($res_array['items']) ? $res_array['items'] : [];

    if (count($res_array_items) === 0 || is_wp_error($response) || wp_remote_retrieve_response_code($response) >= 400) {
        if (file_exists($jsonFilePath)) {
            $jsonData = file_get_contents($jsonFilePath);
            $dataObj = json_decode($jsonData);
            $dataArray = (array) $dataObj;
            return $dataArray;
        } else {
            return array();
        }
    }

    $res_array_data = $res_array_items[0];
    $jsonData = json_encode($res_array_data);
    file_put_contents($jsonFilePath, $jsonData);
    return $res_array_data;
}


function api_data_shortcode_handynummern($atts)
{
    $api_data = impressum09_api_request_telephone();
    $atts = shortcode_atts(
        array(
            'fancy' => 'false',
        ),
        $atts
    );
    // Sicherstellen, dass 'Handynummer' und 'Handynummer_Fancy' vorhanden sind
    $api_data_item_phone = isset($api_data['Handynummer']) ? $api_data['Handynummer'] : '';
    $api_data_item_phone_fancy = isset($api_data['Handynummer_Fancy']) ? $api_data['Handynummer_Fancy'] : '';

    if ($atts['fancy'] === 'true') {
        return $api_data_item_phone_fancy;
    } else {
        return $api_data_item_phone;
    }
}
add_shortcode('handynummer09', 'api_data_shortcode_handynummern');


function api_data_shortcode_callNowButton($atts)
{
    $api_data = impressum09_api_request_telephone();
    $atts = shortcode_atts(
        array(
            'bg-color' => '#333',
            'font-color' => "#fff",
            'link-color' => '#fff'
        ),
        $atts
    );

    $phone_number = isset($api_data["Handynummer"]) ? esc_html($api_data["Handynummer"]) : '';
    $phone_number_fancy = isset($api_data["Handynummer_Fancy"]) ? esc_html($api_data["Handynummer_Fancy"]) : '';

    $formatted_data = "";
    $formatted_data .= "<style>.container{display:none;}@media (max-width:768px){.container{display:block;position:fixed;bottom:0;left:0;width:100%;background-color:" . $atts['bg-color'] . ";color:" . $atts['font-color'] . ";padding:10px;text-align:center;font-weight:bold;font-size:1.5rem}.container a{color:" . $atts['link-color'] . ";text-decoration:none}}</style>";
    $formatted_data .= "<div class=\"container\"><a href=\"tel:" . $phone_number . "\" target=\"_blank\">" . $phone_number_fancy . "</a></div>";

    return $formatted_data;
}
add_shortcode('callNowButton09', 'api_data_shortcode_callNowButton');
